/**
 * Project:  Pro Micro (ATmega32U4) Peripheral Driver Library -- Test Program  
 * ````````
 * File:     main.c
 * `````
 * Author:   M.J.Bauer  [Ref. www.mjbauer.biz]
 * ```````
 * Hardware Platforms:  Sparkfun 'Pro Micro' or Arduino 'Leonardo' board
 * ```````````````````
 * Push-buttons A, B, C, D are wired to port pins PB1, PB2, PB3, PB4, resp.
 * A potentiometer (10k linear) feeds 0..+5V into analog input ADC7.
 *
 * Operating Instructions:
 * ```````````````````````
 * Press button A, B, C or D to select one of 4 PWM outputs to adjust.  
 * Use potentiometer to set PWM duty for the selected output pin.
 *
 * Button A:  Select output OC1A, 10 bit resolution at 16kHz.
 * Button B:  Select output OC1B, 10 bit resolution at 16kHz.
 * Button C:  Select output OC3A, 12 bit resolution at 4kHz.
 * Button D:  Select output OC4D, 10 bit resolution at 16kHz.
 *
 * The 'TX' LED flashes at 2 Hz if the System Timer (TC0) is operating OK.
 *
 * Revision  History:  
 * ````````````````````````````````````````````````````````````````````````````
 *     v1.0 | 04-MAY-2024 | Originated -- MJB
 *     v1.2 | 01-JUL-2024 | Fixed bug in ADC_ReadInput()
 *     v1.3 | 09-SEP-2024 | Improved ADC driver functions
 *
 */
#include <avr/io.h>
#include <stdlib.h>
#include "Pro_Micro_periph_lib.h"  // ATmega32U4 peripheral function library

#define F_CPU  16000000UL    // CPU clock frequency

// When a button is pressed, the corresponding input reads Low (0).
// PORTB is configured to enable pull-ups on the button pins.
#define SETUP_BUTTON_INPUT_PINS()  { DDRB &= 0b11100001;  PORTB |= 0b00011110; }
#define BUTTON_A_PRESSED  ((PINB & (1<<1)) == 0)
#define BUTTON_B_PRESSED  ((PINB & (1<<2)) == 0)
#define BUTTON_C_PRESSED  ((PINB & (1<<3)) == 0)
#define BUTTON_D_PRESSED  ((PINB & (1<<4)) == 0)

#define MODE_A   0  // Test modes
#define MODE_B   1
#define MODE_C   2
#define MODE_D   3

// ------------  Global variables and data  ----------------------------------
unsigned long  TX_LED_period_start;
unsigned  potPosition;  // range 0..1023
unsigned  oc1A_duty, oc1B_duty, oc3A_duty, oc4D_duty;
uint8  testMode;   // One of 3 test options (= 0, 1, or 2)

//============================================================================

void  setup(void)
{
    RX_LED_CONFIGURE();
    TX_LED_CONFIGURE();
    RX_LED_TURN_OFF();
    TX_LED_TURN_ON();
    SETUP_BUTTON_INPUT_PINS();
    
    SET_BIT(DDRB, 5);    // PB5/OC1A set as output 
    SET_BIT(DDRB, 6);    // PB6/OC1B set as output
    SET_BIT(DDRC, 6);    // PC6/OC3A set as output
    SET_BIT(DDRD, 7);    // PD7/OC4D set as output
    
    TC0_Setup_SystemTimer();
    
    TC1_Setup_PWM(1);              // Option 0: PWM 10-bit duty resolution @ 16kHz
    TC3_Setup_PWM(1);              // Option 1: PWM 12-bit duty resolution @ 4kHz
    TC4_Setup_PWM(1);              // Option 1: PWM 10-bit duty resolution @ 16kHz
 
    TC1_OC1A_UpdateDuty(200);      // Initial duty = 20% 
    TC1_OC1B_UpdateDuty(200);      // Initial duty = 20% 
    TC3_OC3A_UpdateDuty(800);      // Initial duty = 20% 
    TC4_OC4D_UpdateDuty(200);      // Initial duty = 20% 
	
	ADC_Setup(ADC_REF_AVCC, ADC_CLOCK_1MHz);
	_delay_ms(2);
    
    GLOBAL_INT_ENABLE();
}
    

void  loop()
{        
    if ((milliseconds() - TX_LED_period_start) >= 50)  
        TX_LED_TURN_OFF();

    if (isTaskPending_50ms())  // Do 50ms periodic task...
    {
        potPosition = ADC_ReadInput(7);  // get pot reading (0..1023)
        if (potPosition > 1000)  potPosition = 1000;  // 0..1000
        
        if (testMode == MODE_A)  oc1A_duty = potPosition;
        else if (testMode == MODE_B)  oc1B_duty = potPosition;
        else if (testMode == MODE_C)  oc3A_duty = potPosition * 4;  // 0..4000
        else  oc4D_duty = potPosition;  // testMode == MODE_D
        
        TC1_OC1A_UpdateDuty(oc1A_duty);
        TC1_OC1B_UpdateDuty(oc1B_duty);
        TC3_OC3A_UpdateDuty(oc3A_duty);
        TC4_OC4D_UpdateDuty(oc4D_duty);
        
        if (testMode != MODE_A && BUTTON_A_PRESSED)  testMode = MODE_A;
        if (testMode != MODE_B && BUTTON_B_PRESSED)  testMode = MODE_B;
        if (testMode != MODE_C && BUTTON_C_PRESSED)  testMode = MODE_C;
        if (testMode != MODE_D && BUTTON_D_PRESSED)  testMode = MODE_D;
    }             
        
    if (isTaskPending_500ms())  // Do 500ms periodic task... 
    {
        TX_LED_TURN_ON();  // Start LED period
        TX_LED_period_start = milliseconds();
    }
}


int  main(void)
{
    setup();
    
    while (TRUE)  loop();   // loop forever
}

// end-of-file
