/**
 * File:  Pro_Micro_periph_lib.c
 *
 * Peripheral function library for projects using the Sparkfun "Pro Micro" module,
 * based on ATmega32U4 MCU.  Also works for the Arduino 'Leonardo' dev board.
 *
 * Originated: April 2024  M.J.Bauer  [www.mjbauer.biz]
 */
#include "Pro_Micro_periph_lib.h"

// ==================  S Y S T E M   T I M E R   F U N C T I O N S  =====================
//
static unsigned long  count_millisecs;  // for function milliseconds()
static volatile bool  TaskFlag_5ms;     // Flag raised every 5 ms
static volatile bool  TaskFlag_50ms;    // Flag raised every 50 ms
static volatile bool  TaskFlag_200ms;   // Flag raised every 200 ms
static volatile bool  TaskFlag_500ms;   // Flag raised every 500 ms
/* 
 * Function:  TC0_Setup_SystemTimer()
 *
 * This function initializes Timer-Counter TC0 to generate a periodic interrupt
 * request (IRQ) once every millisecond, precisely.
 */
void  TC0_Setup_SystemTimer()
{
    TCCR0A = 0b00000010;        // 'CTC' mode enabled;  no output pin.
    TCCR0B = 0x03;              // Prescaler = F_CPU / 64  (Fclk = 250kHz)
    OCR0A = 249;                // Counter cycle = 250 timer clocks = 1ms
    TC0_OCA_IRQ_ENABLE();       // Enable interrupt on Output Compare A match
}

/*
 * Function:  milliseconds()
 *
 * This function returns the value of a free-running 32-bit counter variable,
 * incremented every millisecond by Timer TC0 interrupt handler (ISR).
 * Its purpose is to implement "non-blocking" time delays and event timers.
 */
unsigned long milliseconds()
{
    unsigned long temp32bits;

    // Disable TC0 interrupt to prevent corruption of count_millisecs in case
    // interrupted here in the middle of copying the 4 bytes (32 bits)...
    TC0_OCA_IRQ_DISABLE();

    temp32bits = count_millisecs;  // capture the count value (4 bytes)

    // Re-enable TC1 interrupt
    TC0_OCA_IRQ_ENABLE();

    return  temp32bits;
}

/*
 * Timer-Counter TC0 Interrupt Service Routine (ISR)
 *
 * The timer will generate an IRQ, hence this routine will be executed, when the
 * timer value (TMR0) reaches the 'Top Count' (= OCR0A register value).
 * The timer will be reset (TMR0 = 0) automatically when this occurs.
 */
ISR( TIMER0_COMPA_vect )
{
    static uint8  count_to_5;
    static uint8  count_to_50;
    static uint8  count_to_200;
    static unsigned  count_to_500;
 
    count_millisecs++;   // for milliseconds() function

    // Periodic task scheduler --
    if (++count_to_5 >= 5)  { TaskFlag_5ms = 1;  count_to_5 = 0; }
    if (++count_to_50 >= 50)  { TaskFlag_50ms = 1;  count_to_50 = 0; }
    if (++count_to_200 >= 200)  { TaskFlag_200ms = 1;  count_to_200 = 0; }
    if (++count_to_500 >= 500)  { TaskFlag_500ms = 1;  count_to_500 = 0; }
}

/*
 * Functions:  isTaskPending_??ms()
 *
 * These functions return TRUE if their respective periodic Task Flag is raised;
 * otherwise they return FALSE.  The Task Flag is cleared before the function exits,
 * so that on subsequent calls it will return FALSE, until the next task period ends.
 */
bool  isTaskPending_5ms()
{
    bool  result = TaskFlag_5ms;

    if (result) TaskFlag_5ms = 0;
    return  result;
}

bool  isTaskPending_50ms()
{
    bool  result = TaskFlag_50ms;

    if (result) TaskFlag_50ms = 0;
    return  result;
}

bool  isTaskPending_200ms()
{
    bool  result = TaskFlag_200ms;

    if (result) TaskFlag_200ms = 0;
    return  result;
}

bool  isTaskPending_500ms()
{
    bool  result = TaskFlag_500ms;

    if (result) TaskFlag_500ms = 0;
    return  result;
}


// ====================  U S E R   T I M E R   F U N C T I O N S  =======================
// 
static unsigned  tc3_TOP_count;   // For TC3 period update (unit = TC3 clock)
/*
 * Function:  TC1_Setup_PWM()
 *
 * This function initializes Timer-Counter TC1 in PWM mode to generate two independent 
 * variable-duty pulse waveforms on pins OC1A and OC1B.  The PWM "carrier" frequency and
 * duty resolution are the same on each of the 2 'output compare' channels (A and B).
 *
 * Entry arg:  option -- number specifying timer configuration, as follows:
 *
 *   Option 0:   9-bit duty resolution, 32kHz carrier freq, Tclk = 0.0625 usec
 *   Option 1:  10-bit duty resolution, 16kHz carrier freq, Tclk = 0.0625 usec
 *   Option 2:  12-bit duty resolution, 4kHz  carrier freq, Tclk = 0.0625 usec
 *   Option 3:  14-bit duty resolution, 1kHz  carrier freq, Tclk = 0.0625 usec
 */
void  TC1_Setup_PWM(uint8 option)
{
    TCCR1B = 0;  // Disable timer/counter during setup (CS = 000)
    TCCR1C = 0;  // FOC[2:0] = 0b000
    OCR1A = 0;   // Initial duty = 0
    OCR1B = 0;   // Initial duty = 0
    
    if (option == 0) ICR1 = 499;  // 9 bit PWM @ 32kHz;  TOP count = 499
    if (option == 1) ICR1 = 999;  // 10 bit PWM @ 16kHz;  TOP count = 999
    if (option == 2) ICR1 = 3999;  // 12 bit PWM @ 4kHz;  TOP count = 3999
    if (option == 3) ICR1 = 15999;  // 14 bit PWM @ 1kHz;  TOP count = 15999
    
    // All options: WGM = 1110 (mode 14);  TOP = ICR1
    // The timer/counter is enabled by writing TCCR1B with CS[2:0] != 0
    TCCR1A = 0b10100010;  // COM1A = COM1B = 0b10;  WGM[1:0] = 0b10
    TCCR1B = 0b00011001;  // WGM[3:2] = 11;  CS[2:0] = 001 (no prescaler)
}

/*
 * Function:  TC1_OC1A_Update_Duty() | TC1_OC1B_Update_Duty()
 *
 * Set PWM duty on pin OC1A | OC1B.
 *
 * Entry arg:  duty_clks = duty value, unit = TC1 clock
 *
 * Example:  If the duty resolution is 10 bits, the range of duty_clks is 0..998.
 *           At 50% duty, duty_clks = 500;  at 99.9% duty, duty_clks = 998.
 *           <!> Do not set duty_clks = TOP count, i.e. 999 in this example.
 */
void  TC1_OC1A_UpdateDuty(unsigned duty_clks)
{
    OCR1A = duty_clks;
}


void  TC1_OC1B_UpdateDuty(unsigned duty_clks)
{
    OCR1B = duty_clks;
}


/*
 * Function:  TC3_Setup_PWM()
 *
 * This function initializes Timer-Counter TC3 in PWM mode to generate a variable-duty
 * pulse waveform on pin OC3A.  
 *
 * Entry arg:  option -- number specifying timer configuration, as follows:
 *
 *   Option 0:  10-bit duty resolution, 16kHz carrier freq, Tclk = 0.0625 usec
 *   Option 1:  12-bit duty resolution, 4kHz  carrier freq, Tclk = 0.0625 usec
 */
void  TC3_Setup_PWM(uint8 option)
{
    TCCR3B = 0;  // Disable timer/counter during setup (CS = 000)
    TCCR3C = 0;  // FOC[2:0] = 0b000
    OCR3A = 0;   // Initial duty = 0
    
    if (option == 0)  ICR3 = 999;  // 10 bit PWM @ 16kHz;  TOP = 999
    else  ICR3 = 3999;  // 12 bit PWM @ 4kHz;  TOP count = 3999
    
    // WGM = 1110 (mode 14);  TOP count = ICR3
    // The timer/counter is enabled by writing TCCR3B with CS[2:0] != 0
    TCCR3A = 0b10000010;  // COM3A = 0b10;  WGM[1:0] = 0b10
    TCCR3B = 0b00011001;  // WGM[3:2] = 11;  CS[2:0] = 001 (no prescaler);
}

/*
 * Function:  TC3_OC3A_Update_Duty()
 *
 * Set PWM duty on pin OC3A. 
 *
 * Entry arg:  duty_clks = duty value, unit = TC1 clock 
 *
 * Example:  If the duty resolution is 12 bits, the range of duty_clks is 0..3998.
 *           At 50% duty, duty_clks = 2000;  at 99.9% duty, duty_clks = 3998.
 *           <!> Do not set duty_clks = TOP count, i.e. 3999 in this example.
 */
void  TC3_OC3A_UpdateDuty(unsigned duty_clks)
{
    OCR3A = duty_clks;
}

/*
 * Function:  TC3_Setup_PulseGen()
 *
 * This function initializes Timer-Counter TC3 in pulse generator mode with variable
 * frequency and fixed pulse-width output on pin OC3A. The pulse width (duty) may be
 * changed anytime by calling function TC3_OC3A_UpdateDuty().
 *
 * Entry args:  1. polarity  -- To set output polarity (0: High-going, 1: Low-going pulse)
 *              2. prescaler -- Prescaler value, 3 bit value written to TCCR3B bits[2:0]
 *                 Timer Fclk = CPU Fosc / N;  Timer Tclk = CPU Tosc x N
 *                 (prescaler = 1: N = 1;  2: N = 8;  3: N = 64;  4: N = 256;  5: N = 1024)
 *              3. duty_clks -- Initial pulse width (duty), unit = Timer clock period.
 */
void  TC3_Setup_PulseGen(uint8 polarity, uint8 prescaler, unsigned duty_clks)
{
    TCCR3B = 0;              // Disable timer/counter during setup (CS = 000)
    TCCR3C = 0;              // FOC[2:0] = 0b000
    OCR3A = duty_clks;       // Set initial (fixed) duty
    ICR3 = duty_clks * 2;    // Initial TOP count = duty * 2
    
    if (polarity == 0) TCCR3A = 0b10000010;  // COM1A = 0b10;  WGM[1:0] = 0b10 (mode 14)
    else  TCCR3A = 0b11000010;  // COM1A = 0b11;  WGM[1:0] = 0b10 (mode 14)

    // The timer/counter will be enabled by writing TCCR3B prescaler bits CS[2:0]
    TCCR3B = 0b00011000;     // WGM[3:2] = 11 (mode 14)
    TCCR3B |= prescaler;     // CS[2:0] = prescaler
    TC3_OVF_IRQ_ENABLE();    // Enable interrupt for TC3 overflow (at TOP)
}    

/*
 * Function:  TC3_OC3A_UpdatePeriod()
 *
 * Set output pulse period on pin OC3A. 
 *
 * The period register (ICR3) is written on the next TC3 overflow interrupt. Register
 * ICR3 update must be synchronized to the timer overflow because ICR3 is not double-
 * buffered. (The OC registers are double-buffered, so duty may be updated anytime.)
 *
 * Entry arg:  period_clks = output pulse period, unit = TC3 clock period
 *
 * Note:  Arg1 (period_clks) must be greater than the duty value (duty_clks).
 */
void  TC3_OC3A_UpdatePeriod(unsigned period_clks)
{
    tc3_TOP_count = period_clks;
}

/*
 * Timer-Counter TC3 Interrupt Service Routine...
 * IRQ occurs at Timer Overflow at TOP count (= ICR3).
 */
ISR( TIMER3_OVF_vect )
{
    ICR3 = tc3_TOP_count;
}


/*
 * Function:  TC4_Setup_PWM()
 *
 * This function initializes Timer-Counter TC4 in fast PWM mode to generate a variable-duty
 * pulse waveform on pin PD7/OC4D.
 *
 * Entry arg:  option -- number specifying timer configuration, as follows:
 *
 *   Option 0:   8-bit duty resolution, 32kHz carrier freq, Tclk = 0.125 usec
 *   Option 1:  10-bit duty resolution, 16kHz carrier freq, Tclk = 0.0625 usec
 */
void  TC4_Setup_PWM(uint8 option)
{
    TCCR4A = 0;  // Disable timer/counter during setup (CS = 0000)
    TCCR4B = 0;
    TCCR4C = 0;
    TCCR4D = 0;
    TCCR4E = 0;
    TC4H = 0;
    OCR4D  = 0;  // Initial duty = 0
    
    // The timer/counter is enabled by writing TCCR4B with CS[3:0] != 0
    if (option == 0)   // 8 bit PWM @ 32kHz
    {
        OCR4C = 249;          // TOP count
        TCCR4A = 0b00000000;  // COM4A = 0, COM4B = 0  (OCRA, OCRB not used)
        TCCR4C = 0b00001001;  // COM4D[1:0] = 0b10 (high-going pulseduty), PWM4D = 1
        TCCR4D = 0b00000000;  // WGM4[1:0] = 0b00 (fast PWM mode)
        TCCR4B = 0b00000010;  // Clock select: F_CPU/2 (8 MHz)
    }    
    else   // 10 bit PWM @ 16kHz
    {
        TC4H  = HI_BYTE(999);  // TOP count 2 MS bits
        OCR4C = LO_BYTE(999);  // TOP count 8 LS bits
        TCCR4A = 0b00000000;    // COM4A = 0, COM4B = 0  (OCRA, OCRB not used)
        TCCR4C = 0b00001001;    // COM4D[1:0] = 0b10 (high-going pulseduty), PWM4D = 1
        TCCR4D = 0b00000000;    // WGM4[1:0] = 0b00 (fast PWM mode)
        TCCR4B = 0b00000001;    // Clock select: F_CPU (16MHz)
    }    
}    

/*
 * Function:  TC4_OC4D_Update_Duty()
 *
 * Set PWM duty on pin OC4D. 
 *
 * Entry arg:  duty_clks = duty value, unit = TC4 clock source
 *
 * Example:  If the duty resolution is 10 bits, the range of duty_clks is 0..999.
 *           At 50% duty, duty_clks = 500;  at 100% duty, duty_clks = 999.
 */
void  TC4_OC4D_UpdateDuty(unsigned duty_clks)
{
    TC4H  = HI_BYTE(duty_clks);  // OCR4D 2 MS bits
    OCR4D = LO_BYTE(duty_clks);  // OCR4D 8 LS bits
}


//==========================   A D C   F U N C T I O N S  ===============================
/*
 * Function ADC_Setup() initializes the ADC to use a specified reference source
 * and clock frequency (derived from the system clock, assumed F_CPU = 16MHz).
 * On exit, the ADC is enabled. 
 * <!> A delay of 50us (minimum) is required after switching the ADC reference source
 * to allow the reference to stabilize.
 *
 * Entry args:  opt_ref = 0: AREF, 1: AVCC, 2: N/A, 3: Internal 2.56V 
 *              opt_clock = 0: N/A, 2: 4MHz, 3: 2MHz, 4: 1MHz, 5: 500kHz, 6: 250kHz
 */
void  ADC_Setup(uint8 opt_ref, uint8 opt_clock)
{
    ADMUX = opt_ref << 6;     // ADMUX[7:6] = Ref bits
    if (opt_clock >= 2 && opt_clock <= 6) ADCSRA = opt_clock;
    else  ADCSRA = 5;         // default
    SET_BIT(ADCSRA, ADEN);    // Enable ADC
}
    
/*
 * Function ADC_ReadInput() starts a one-off conversion on the given input, waits for
 * the conversion cycle to complete, then returns the 10-bit result.
 * The function ADC_Setup() must be called prior, once at start-up and again each time
 * the ADC reference or clock rate is changed.
 * The first conversion after switching ADC input/channel may be inaccurate.
 *
 * Entry arg:  muxsel = MUX[5:0] = ADC MUX input (channel) select:  
 *              0 = ADC0, 1 = ADC1, 4 = ADC4, 5 = ADC5, 6 = ADC6, 7 = ADC7
 *             30 = bandgap ref (1.1V),  31 = GND
 *             32 = ADC8, 33 = ADC9, 34 = ADC10, 35 = ADC11, 36 = ADC12, 37 = ADC13
 *             39 = temperature sensor
 */
unsigned  ADC_ReadInput(uint8 muxsel)
{
    ADCSRB = 0x00;
    ADMUX &= 0xC0;  // preserve reference select bits [7:6]
    ADMUX |= (muxsel & 0x1F);  // add 5 LS bits = MUX[4:0]
    if (muxsel & (1<<5)) ADCSRB = 0x20;  // set MUX5 bit in ADCSRB
    SET_BIT(ADCSRA, ADSC);  // Start conversion

    while (TEST_BIT(ADCSRA, ADSC) != 0)  { ; }  // wait till conversion done

    return  ADC;
}


//========================   E E P R O M    F U N C T I O N S  ==========================
//
// See header file (pro-micro-periph.h) for usage
//
uint8  EEPROM_ReadByte(unsigned address)
{
    uint8  bDat;  // returned data
    uint8  int_status = SREG & 0x80;   // save I flag from SREG

    CLEAR_BIT(SREG, 7);       // global interrupt disable

    while (TEST_BIT(EECR, EEPE) != 0)
    {
        ;  // Wait for completion of previous write
    }

    EEAR = address;           // Set up address register
    SET_BIT(EECR, EERE);      // Start eeprom read
    bDat = EEDR;              // read out the data reg.

    SREG |= int_status;       // restore caller's interrupt status
    return  bDat;
}

void  EEPROM_WriteByte(unsigned address, uint8 bDat)
{
    uint8  int_status = SREG & 0x80;   // save I flag from SREG

    CLEAR_BIT(SREG, 7);       // global interrupt disable

    while (TEST_BIT(EECR, EEPE) != 0)
    {
        ;  // Wait for completion of previous write
    }

    EEAR = address;           // Set up address register
    EEDR = bDat;              // Set up the data reg.
    SET_BIT(EECR, EEMPE);     // Master write enable
    SET_BIT(EECR, EEPE);      // Start eeprom write

    SREG |= int_status;       // restore caller's interrupt status
}

void  EEPROM_ReadArray(uint8 *pdat, int nbytes)
{
    unsigned address = 0;

    while (nbytes != 0)
    {
        *pdat++ = EEPROM_ReadByte(address);
        address++;
        nbytes--;
    }
}

void  EEPROM_WriteArray(uint8 *pdat, int nbytes)
{
    unsigned address = 0;

    while (nbytes != 0)
    {
        EEPROM_WriteByte(address, *pdat);
        address++;
        pdat++;
        nbytes--;
    }
}


//==========   F L A S H   M E M O R Y   D A T A   R E A D   F U N C T I O N   ==========
//
// See header file (pro-micro-periph.h) for usage
//
void  PGM_ReadData(const uint8 *src, uint8 *dest, unsigned nbytes)
{
    while (nbytes != 0)
    {
        *dest++ = pgm_read_byte(src++);
        --nbytes;
    }
}


//==========================   U A R T    F U N C T I O N S  ============================
//
// If user application requires UART data reception to be interrupt-driven, the program
// must define a "call-back" function named UART_RX_IRQ_Handler() to read received data
// bytes from the UART and store the RX data in a (circular) FIFO buffer in MCU SRAM.
// Symbol 'UART_RX_INTERRUPT_DRIVEN' must be defined in header file "pro-micro-periph.h"
// and the library code must be re-compiled (rebuilt) to enable this option.
//
void  UART_init(unsigned baudrate)
{
    unsigned brr = F_CPU / ((long)baudrate * 16) - 1;

    UBRR1H = HI_BYTE(brr);      // set the Baud rate
    UBRR1L = LO_BYTE(brr);

    SET_BIT(UCSR1C, UCSZ10);    // Set frame format: 8,N,1 (data,parity,stop)
    SET_BIT(UCSR1B, RXEN1);     // Enable Receiver
    SET_BIT(UCSR1B, TXEN1);     // Enable Transmitter
}

uint8  UART_GetByte(void)
{
    unsigned char  b = 0;

    if (TEST_BIT(UCSR1A, RXC1) != 0)  b = UDR1;  // RX data available

    return  b;
}

void  UART_PutByte(unsigned char b)
{
    while (TEST_BIT(UCSR1A, UDRE1) == 0) { ;;; }
        
    UDR1 = b;
}

void  UART_PutString(char *str)
{
    char c;

    while ( (c = *str++) != '\0' )
    {
        if ( c == '\n' )
        {
            UART_PutByte( '\r' );
            UART_PutByte( '\n' );
        }
        else   UART_PutByte( c );
    }
}

/*---------------------------------------------------------------------------------------
*   Name:       UART_PutDecimal
*
*   Function:   Outputs a 16-bit word as an unsigned decimal integer, up to 5 digits,
*               right justified in the specified field, padded with leading zeros.
*               If the value is too big to fit into the specified minimum field size,
*               the field will be expanded to accommodate the number of digits.
*
*   Args:       uw = unsigned word to be converted and output
*               fieldSize = minimum number of character places to output (1..5)
*
*--------------------------------------------------------------------------------------*/
void  UART_PutDecimal(uint16 uw, uint8 fieldSize)
{
    uint8   digit[6];     // digit[0] is LSD
    uint8   place, dig;
    bool    isLeading0 = TRUE;

    if ( fieldSize > 5 )  fieldSize = 5;
    if ( fieldSize < 1 )  fieldSize = 1;

    for ( place = 0;  place < 5;  place++ )   // begin conversion with LSD
    {
        digit[place] = (uw % 10);
        uw = uw / 10;
    }

    for ( place = 4;  place < 5;  place-- )   // begin output with MSD
    {
        dig = digit[place];
        if (dig != 0)  isLeading0 = FALSE;           // Found non-zero digit
        if (place < fieldSize)  UART_PutDigit(dig);  // Inside minimum field... output always,
        else if (!isLeading0)  UART_PutDigit(dig);   // else... output only if NOT leading zero
    }
}

/*-------------------------------------------------------------------------------------
 * Name               :  UART_PutDigit()
 * Function           :  Show hex/decimal digit value (1 char)
 *
 * Input              :  bDat = nybble to be converted and output
 * Return             :  --
 --------------------------------------------------------------------------------------*/
void   UART_PutDigit(uint8 bDat)
{
    bDat &= 0x0F;
    if (bDat < 10) UART_PutChar('0' + bDat);
    else  UART_PutChar('A' + bDat - 10);
}

#ifdef UART_RX_INTERRUPT_DRIVEN
/*
*   UART0 receiver interrupt service routine.
*
*   The IRQ signals that one or more bytes have been received by the UART;
*   the byte(s) must be read out of the UART RX data register(s) and stored in a
*   buffer in RAM (circular FIFO) by the call-back function: UART_RX_IRQ_Handler().
*/
ISR( USART_RX_vect )
{
    if (TEST_BIT(UCSR1A, RXC1) != 0)    // RX data available
    {
        UART_RX_IRQ_Handler();
    }
    else
    {
        // Clear source of unexpected IRQ (todo)
    }
}
#endif // UART_RX_INTERRUPT_DRIVEN

// end-of-file
